// Copyright 2025 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package merchantapi provides access to the Merchant API.
//
// For product documentation, see: https://developers.devsite.corp.google.com/merchant/api
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/merchantapi/products_v1beta"
//	...
//	ctx := context.Background()
//	merchantapiService, err := merchantapi.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	merchantapiService, err := merchantapi.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	merchantapiService, err := merchantapi.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package merchantapi // import "google.golang.org/api/merchantapi/products_v1beta"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "merchantapi:products_v1beta"
const apiName = "merchantapi"
const apiVersion = "products_v1beta"
const basePath = "https://merchantapi.googleapis.com/"
const basePathTemplate = "https://merchantapi.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://merchantapi.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// Manage your product listings and accounts for Google Shopping
	ContentScope = "https://www.googleapis.com/auth/content"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/content",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &Service{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.Accounts = NewAccountsService(s)
	if err != nil {
		return nil, err
	}
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.Background(), option.WithHTTPClient(client))
}

type Service struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Accounts *AccountsService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewAccountsService(s *Service) *AccountsService {
	rs := &AccountsService{s: s}
	rs.ProductInputs = NewAccountsProductInputsService(s)
	rs.Products = NewAccountsProductsService(s)
	return rs
}

type AccountsService struct {
	s *Service

	ProductInputs *AccountsProductInputsService

	Products *AccountsProductsService
}

func NewAccountsProductInputsService(s *Service) *AccountsProductInputsService {
	rs := &AccountsProductInputsService{s: s}
	return rs
}

type AccountsProductInputsService struct {
	s *Service
}

func NewAccountsProductsService(s *Service) *AccountsProductsService {
	rs := &AccountsProductsService{s: s}
	return rs
}

type AccountsProductsService struct {
	s *Service
}

// Attributes: Attributes.
type Attributes struct {
	// AdditionalImageLinks: Additional URLs of images of the item.
	AdditionalImageLinks []string `json:"additionalImageLinks,omitempty"`
	// AdsGrouping: Used to group items in an arbitrary way. Only for CPA%,
	// discouraged otherwise.
	AdsGrouping string `json:"adsGrouping,omitempty"`
	// AdsLabels: Similar to ads_grouping, but only works on CPC.
	AdsLabels []string `json:"adsLabels,omitempty"`
	// AdsRedirect: Allows advertisers to override the item URL when the product is
	// shown within the context of Product ads.
	AdsRedirect string `json:"adsRedirect,omitempty"`
	// Adult: Set to true if the item is targeted towards adults.
	Adult bool `json:"adult,omitempty"`
	// AgeGroup: Target age group
	// (https://support.google.com/merchants/answer/6324463) of the item.
	AgeGroup string `json:"ageGroup,omitempty"`
	// AutoPricingMinPrice: A safeguard in the "Automated Discounts"
	// (https://support.google.com/merchants/answer/10295759) and "Dynamic
	// Promotions" (https://support.google.com/merchants/answer/13949249) projects,
	// ensuring that discounts on merchants' offers do not fall below this value,
	// thereby preserving the offer's value and profitability.
	AutoPricingMinPrice *Price `json:"autoPricingMinPrice,omitempty"`
	// Availability: Availability status of the item.
	Availability string `json:"availability,omitempty"`
	// AvailabilityDate: The day a pre-ordered product becomes available for
	// delivery, in ISO 8601 (http://en.wikipedia.org/wiki/ISO_8601) format.
	AvailabilityDate string `json:"availabilityDate,omitempty"`
	// Brand: Brand of the item.
	Brand string `json:"brand,omitempty"`
	// CanonicalLink: URL for the canonical version of your item's landing page.
	CanonicalLink string `json:"canonicalLink,omitempty"`
	// Certifications: Product Certifications, for example for energy efficiency
	// labeling of products recorded in the EU EPREL
	// (https://eprel.ec.europa.eu/screen/home) database. See the Help Center
	// (https://support.google.com/merchants/answer/13528839) article for more
	// information.
	Certifications []*Certification `json:"certifications,omitempty"`
	// CloudExportAdditionalProperties: Extra fields to export to the Cloud Retail
	// program.
	CloudExportAdditionalProperties []*CloudExportAdditionalProperties `json:"cloudExportAdditionalProperties,omitempty"`
	// Color: Color of the item.
	Color string `json:"color,omitempty"`
	// Condition: Condition or state of the item.
	Condition string `json:"condition,omitempty"`
	// CostOfGoodsSold: Cost of goods sold. Used for gross profit reporting.
	CostOfGoodsSold *Price `json:"costOfGoodsSold,omitempty"`
	// CustomLabel0: Custom label 0 for custom grouping of items in a Shopping
	// campaign.
	CustomLabel0 string `json:"customLabel0,omitempty"`
	// CustomLabel1: Custom label 1 for custom grouping of items in a Shopping
	// campaign.
	CustomLabel1 string `json:"customLabel1,omitempty"`
	// CustomLabel2: Custom label 2 for custom grouping of items in a Shopping
	// campaign.
	CustomLabel2 string `json:"customLabel2,omitempty"`
	// CustomLabel3: Custom label 3 for custom grouping of items in a Shopping
	// campaign.
	CustomLabel3 string `json:"customLabel3,omitempty"`
	// CustomLabel4: Custom label 4 for custom grouping of items in a Shopping
	// campaign.
	CustomLabel4 string `json:"customLabel4,omitempty"`
	// Description: Description of the item.
	Description string `json:"description,omitempty"`
	// DisclosureDate: The date time when an offer becomes visible in search
	// results across Google’s YouTube surfaces, in ISO 8601
	// (http://en.wikipedia.org/wiki/ISO_8601) format. See Disclosure date (
	// https://support.google.com/merchants/answer/13034208) for more information.
	DisclosureDate string `json:"disclosureDate,omitempty"`
	// DisplayAdsId: An identifier for an item for dynamic remarketing campaigns.
	DisplayAdsId string `json:"displayAdsId,omitempty"`
	// DisplayAdsLink: URL directly to your item's landing page for dynamic
	// remarketing campaigns.
	DisplayAdsLink string `json:"displayAdsLink,omitempty"`
	// DisplayAdsSimilarIds: Advertiser-specified recommendations.
	DisplayAdsSimilarIds []string `json:"displayAdsSimilarIds,omitempty"`
	// DisplayAdsTitle: Title of an item for dynamic remarketing campaigns.
	DisplayAdsTitle string `json:"displayAdsTitle,omitempty"`
	// DisplayAdsValue: Offer margin for dynamic remarketing campaigns.
	DisplayAdsValue float64 `json:"displayAdsValue,omitempty"`
	// EnergyEfficiencyClass: The energy efficiency class as defined in EU
	// directive 2010/30/EU.
	EnergyEfficiencyClass string `json:"energyEfficiencyClass,omitempty"`
	// ExcludedDestinations: The list of destinations to exclude for this target
	// (corresponds to unchecked check boxes in Merchant Center).
	ExcludedDestinations []string `json:"excludedDestinations,omitempty"`
	// ExpirationDate: Date on which the item should expire, as specified upon
	// insertion, in ISO 8601 (http://en.wikipedia.org/wiki/ISO_8601) format. The
	// actual expiration date is exposed in `productstatuses` as
	// googleExpirationDate (https://support.google.com/merchants/answer/6324499)
	// and might be earlier if `expirationDate` is too far in the future.
	ExpirationDate string `json:"expirationDate,omitempty"`
	// ExternalSellerId: Required for multi-seller accounts. Use this attribute if
	// you're a marketplace uploading products for various sellers to your
	// multi-seller account.
	ExternalSellerId string `json:"externalSellerId,omitempty"`
	// FreeShippingThreshold: Conditions to be met for a product to have free
	// shipping.
	FreeShippingThreshold []*FreeShippingThreshold `json:"freeShippingThreshold,omitempty"`
	// Gender: Target gender of the item.
	Gender string `json:"gender,omitempty"`
	// GoogleProductCategory: Google's category of the item (see Google product
	// taxonomy (https://support.google.com/merchants/answer/1705911)). When
	// querying products, this field will contain the user provided value. There is
	// currently no way to get back the auto assigned google product categories
	// through the API.
	GoogleProductCategory string `json:"googleProductCategory,omitempty"`
	// Gtin: Global Trade Item Numbers (GTIN
	// (https://support.google.com/merchants/answer/188494#gtin)) of the item. You
	// can provide up to 10 GTINs.
	Gtin []string `json:"gtin,omitempty"`
	// IdentifierExists: Set this value to false when the item does not have unique
	// product identifiers appropriate to its category, such as GTIN, MPN, and
	// brand. Defaults to true, if not provided.
	IdentifierExists bool `json:"identifierExists,omitempty"`
	// ImageLink: URL of an image of the item.
	ImageLink string `json:"imageLink,omitempty"`
	// IncludedDestinations: The list of destinations to include for this target
	// (corresponds to checked check boxes in Merchant Center). Default
	// destinations are always included unless provided in `excludedDestinations`.
	IncludedDestinations []string `json:"includedDestinations,omitempty"`
	// Installment: Number and amount of installments to pay for an item.
	Installment *Installment `json:"installment,omitempty"`
	// IsBundle: Whether the item is a merchant-defined bundle. A bundle is a
	// custom grouping of different products sold by a merchant for a single price.
	IsBundle bool `json:"isBundle,omitempty"`
	// ItemGroupId: Shared identifier for all variants of the same product.
	ItemGroupId string `json:"itemGroupId,omitempty"`
	// LifestyleImageLinks: Additional URLs of lifestyle images of the item, used
	// to explicitly identify images that showcase your item in a real-world
	// context. See the Help Center article
	// (https://support.google.com/merchants/answer/9103186) for more information.
	LifestyleImageLinks []string `json:"lifestyleImageLinks,omitempty"`
	// Link: URL directly linking to your item's page on your online store.
	Link string `json:"link,omitempty"`
	// LinkTemplate: Link template for merchant hosted local storefront.
	LinkTemplate string `json:"linkTemplate,omitempty"`
	// LoyaltyPoints: Loyalty points that users receive after purchasing the item.
	// Japan only.
	LoyaltyPoints *LoyaltyPoints `json:"loyaltyPoints,omitempty"`
	// LoyaltyPrograms: A list of loyalty program information that is used to
	// surface loyalty benefits (for example, better pricing, points, etc) to the
	// user of this item.
	LoyaltyPrograms []*LoyaltyProgram `json:"loyaltyPrograms,omitempty"`
	// Material: The material of which the item is made.
	Material string `json:"material,omitempty"`
	// MaxEnergyEfficiencyClass: The energy efficiency class as defined in EU
	// directive 2010/30/EU.
	MaxEnergyEfficiencyClass string `json:"maxEnergyEfficiencyClass,omitempty"`
	// MaxHandlingTime: Maximal product handling time (in business days).
	MaxHandlingTime int64 `json:"maxHandlingTime,omitempty,string"`
	// MinEnergyEfficiencyClass: The energy efficiency class as defined in EU
	// directive 2010/30/EU.
	MinEnergyEfficiencyClass string `json:"minEnergyEfficiencyClass,omitempty"`
	// MinHandlingTime: Minimal product handling time (in business days).
	MinHandlingTime int64 `json:"minHandlingTime,omitempty,string"`
	// MobileLink: URL for the mobile-optimized version of your item's landing
	// page.
	MobileLink string `json:"mobileLink,omitempty"`
	// MobileLinkTemplate: Link template for merchant hosted local storefront
	// optimized for mobile devices.
	MobileLinkTemplate string `json:"mobileLinkTemplate,omitempty"`
	// Mpn: Manufacturer Part Number (MPN
	// (https://support.google.com/merchants/answer/188494#mpn)) of the item.
	Mpn string `json:"mpn,omitempty"`
	// Multipack: The number of identical products in a merchant-defined multipack.
	Multipack int64 `json:"multipack,omitempty,string"`
	// Pattern: The item's pattern (for example, polka dots).
	Pattern string `json:"pattern,omitempty"`
	// Pause: Publication of this item will be temporarily paused
	// (https://support.google.com/merchants/answer/11909930).
	Pause string `json:"pause,omitempty"`
	// PickupMethod: The pick up option for the item.
	PickupMethod string `json:"pickupMethod,omitempty"`
	// PickupSla: Item store pickup timeline.
	PickupSla string `json:"pickupSla,omitempty"`
	// Price: Price of the item.
	Price *Price `json:"price,omitempty"`
	// ProductDetails: Technical specification or additional product details.
	ProductDetails []*ProductDetail `json:"productDetails,omitempty"`
	// ProductHeight: The height of the product in the units provided. The value
	// must be between 0 (exclusive) and 3000 (inclusive).
	ProductHeight *ProductDimension `json:"productHeight,omitempty"`
	// ProductHighlights: Bullet points describing the most relevant highlights of
	// a product.
	ProductHighlights []string `json:"productHighlights,omitempty"`
	// ProductLength: The length of the product in the units provided. The value
	// must be between 0 (exclusive) and 3000 (inclusive).
	ProductLength *ProductDimension `json:"productLength,omitempty"`
	// ProductTypes: Categories of the item (formatted as in product data
	// specification
	// (https://support.google.com/merchants/answer/188494#product_type)).
	ProductTypes []string `json:"productTypes,omitempty"`
	// ProductWeight: The weight of the product in the units provided. The value
	// must be between 0 (exclusive) and 2000 (inclusive).
	ProductWeight *ProductWeight `json:"productWeight,omitempty"`
	// ProductWidth: The width of the product in the units provided. The value must
	// be between 0 (exclusive) and 3000 (inclusive).
	ProductWidth *ProductDimension `json:"productWidth,omitempty"`
	// PromotionIds: The unique ID of a promotion.
	PromotionIds []string `json:"promotionIds,omitempty"`
	// SalePrice: Advertised sale price of the item.
	SalePrice *Price `json:"salePrice,omitempty"`
	// SalePriceEffectiveDate: Date range during which the item is on sale (see
	// product data specification
	// (https://support.google.com/merchants/answer/188494#sale_price_effective_date)).
	SalePriceEffectiveDate *Interval `json:"salePriceEffectiveDate,omitempty"`
	// SellOnGoogleQuantity: The quantity of the product that is available for
	// selling on Google. Supported only for online products.
	SellOnGoogleQuantity int64 `json:"sellOnGoogleQuantity,omitempty,string"`
	// Shipping: Shipping rules.
	Shipping []*Shipping `json:"shipping,omitempty"`
	// ShippingHeight: Height of the item for shipping.
	ShippingHeight *ShippingDimension `json:"shippingHeight,omitempty"`
	// ShippingLabel: The shipping label of the product, used to group product in
	// account-level shipping rules.
	ShippingLabel string `json:"shippingLabel,omitempty"`
	// ShippingLength: Length of the item for shipping.
	ShippingLength *ShippingDimension `json:"shippingLength,omitempty"`
	// ShippingWeight: Weight of the item for shipping.
	ShippingWeight *ShippingWeight `json:"shippingWeight,omitempty"`
	// ShippingWidth: Width of the item for shipping.
	ShippingWidth *ShippingDimension `json:"shippingWidth,omitempty"`
	// ShoppingAdsExcludedCountries: List of country codes (ISO 3166-1 alpha-2) to
	// exclude the offer from Shopping Ads destination. Countries from this list
	// are removed from countries configured in data source settings.
	ShoppingAdsExcludedCountries []string `json:"shoppingAdsExcludedCountries,omitempty"`
	// Size: Size of the item. Only one value is allowed. For variants with
	// different sizes, insert a separate product for each size with the same
	// `itemGroupId` value (see https://support.google.com/merchants/answer/6324492
	// (size definition)).
	Size string `json:"size,omitempty"`
	// SizeSystem: System in which the size is specified. Recommended for apparel
	// items.
	SizeSystem string `json:"sizeSystem,omitempty"`
	// SizeTypes: The cut of the item. It can be used to represent combined size
	// types for apparel items. Maximum two of size types can be provided (see
	// https://support.google.com/merchants/answer/6324497 (size type)).
	SizeTypes []string `json:"sizeTypes,omitempty"`
	// StructuredDescription: Structured description, for algorithmically
	// (AI)-generated descriptions.
	StructuredDescription *ProductStructuredDescription `json:"structuredDescription,omitempty"`
	// StructuredTitle: Structured title, for algorithmically (AI)-generated
	// titles.
	StructuredTitle *ProductStructuredTitle `json:"structuredTitle,omitempty"`
	// SubscriptionCost: Number of periods (months or years) and amount of payment
	// per period for an item with an associated subscription contract.
	SubscriptionCost *SubscriptionCost `json:"subscriptionCost,omitempty"`
	// SustainabilityIncentives: The list of sustainability incentive programs.
	SustainabilityIncentives []*ProductSustainabilityIncentive `json:"sustainabilityIncentives,omitempty"`
	// TaxCategory: The tax category of the product.
	TaxCategory string `json:"taxCategory,omitempty"`
	// Taxes: Tax information.
	Taxes []*Tax `json:"taxes,omitempty"`
	// Title: Title of the item.
	Title string `json:"title,omitempty"`
	// TransitTimeLabel: The transit time label of the product, used to group
	// product in account-level transit time tables.
	TransitTimeLabel string `json:"transitTimeLabel,omitempty"`
	// UnitPricingBaseMeasure: The preference of the denominator of the unit price.
	UnitPricingBaseMeasure *UnitPricingBaseMeasure `json:"unitPricingBaseMeasure,omitempty"`
	// UnitPricingMeasure: The measure and dimension of an item.
	UnitPricingMeasure *UnitPricingMeasure `json:"unitPricingMeasure,omitempty"`
	// VirtualModelLink: URL of the 3D image of the item. See the Help Center
	// article (https://support.google.com/merchants/answer/13674896) for more
	// information.
	VirtualModelLink string `json:"virtualModelLink,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AdditionalImageLinks") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdditionalImageLinks") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Attributes) MarshalJSON() ([]byte, error) {
	type NoMethod Attributes
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *Attributes) UnmarshalJSON(data []byte) error {
	type NoMethod Attributes
	var s1 struct {
		DisplayAdsValue gensupport.JSONFloat64 `json:"displayAdsValue"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.DisplayAdsValue = float64(s1.DisplayAdsValue)
	return nil
}

// Certification: Product certification
// (https://support.google.com/merchants/answer/13528839), initially introduced
// for EU energy efficiency labeling compliance using the EU EPREL database.
type Certification struct {
	// CertificationAuthority: The certification authority, for example
	// "European_Commission". Maximum length is 2000 characters.
	CertificationAuthority string `json:"certificationAuthority,omitempty"`
	// CertificationCode: The certification code. Maximum length is 2000
	// characters.
	CertificationCode string `json:"certificationCode,omitempty"`
	// CertificationName: The name of the certification, for example "EPREL".
	// Maximum length is 2000 characters.
	CertificationName string `json:"certificationName,omitempty"`
	// CertificationValue: The certification value (also known as class, level or
	// grade), for example "A+", "C", "gold". Maximum length is 2000 characters.
	CertificationValue string `json:"certificationValue,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CertificationAuthority") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CertificationAuthority") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Certification) MarshalJSON() ([]byte, error) {
	type NoMethod Certification
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CloudExportAdditionalProperties: Product property for the Cloud Retail API.
// For example, properties for a TV product could be "Screen-Resolution" or
// "Screen-Size".
type CloudExportAdditionalProperties struct {
	// BoolValue: Boolean value of the given property. For example for a TV
	// product, "True" or "False" if the screen is UHD.
	BoolValue bool `json:"boolValue,omitempty"`
	// FloatValue: Float values of the given property. For example for a TV product
	// 1.2345. Maximum repeatedness of this value is 400. Values are stored in an
	// arbitrary but consistent order.
	FloatValue []float64 `json:"floatValue,omitempty"`
	// IntValue: Integer values of the given property. For example, 1080 for a TV
	// product's Screen Resolution. Maximum repeatedness of this value is 400.
	// Values are stored in an arbitrary but consistent order.
	IntValue googleapi.Int64s `json:"intValue,omitempty"`
	// MaxValue: Maximum float value of the given property. For example for a TV
	// product 100.00.
	MaxValue float64 `json:"maxValue,omitempty"`
	// MinValue: Minimum float value of the given property. For example for a TV
	// product 1.00.
	MinValue float64 `json:"minValue,omitempty"`
	// PropertyName: Name of the given property. For example, "Screen-Resolution"
	// for a TV product. Maximum string size is 256 characters.
	PropertyName string `json:"propertyName,omitempty"`
	// TextValue: Text value of the given property. For example, "8K(UHD)" could be
	// a text value for a TV product. Maximum repeatedness of this value is 400.
	// Values are stored in an arbitrary but consistent order. Maximum string size
	// is 256 characters.
	TextValue []string `json:"textValue,omitempty"`
	// UnitCode: Unit of the given property. For example, "Pixels" for a TV
	// product. Maximum string size is 256B.
	UnitCode string `json:"unitCode,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BoolValue") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BoolValue") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CloudExportAdditionalProperties) MarshalJSON() ([]byte, error) {
	type NoMethod CloudExportAdditionalProperties
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *CloudExportAdditionalProperties) UnmarshalJSON(data []byte) error {
	type NoMethod CloudExportAdditionalProperties
	var s1 struct {
		FloatValue []gensupport.JSONFloat64 `json:"floatValue"`
		MaxValue   gensupport.JSONFloat64   `json:"maxValue"`
		MinValue   gensupport.JSONFloat64   `json:"minValue"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.FloatValue = make([]float64, len(s1.FloatValue))
	for i := range s1.FloatValue {
		s.FloatValue[i] = float64(s1.FloatValue[i])
	}
	s.MaxValue = float64(s1.MaxValue)
	s.MinValue = float64(s1.MinValue)
	return nil
}

// CustomAttribute: A message that represents custom attributes. Exactly one of
// `value` or `group_values` must not be empty.
type CustomAttribute struct {
	// GroupValues: Subattributes within this attribute group. If `group_values` is
	// not empty, `value` must be empty.
	GroupValues []*CustomAttribute `json:"groupValues,omitempty"`
	// Name: The name of the attribute.
	Name string `json:"name,omitempty"`
	// Value: The value of the attribute. If `value` is not empty, `group_values`
	// must be empty.
	Value string `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "GroupValues") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "GroupValues") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CustomAttribute) MarshalJSON() ([]byte, error) {
	type NoMethod CustomAttribute
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DestinationStatus: The destination status of the product status.
type DestinationStatus struct {
	// ApprovedCountries: List of country codes (ISO 3166-1 alpha-2) where the
	// offer is approved.
	ApprovedCountries []string `json:"approvedCountries,omitempty"`
	// DisapprovedCountries: List of country codes (ISO 3166-1 alpha-2) where the
	// offer is disapproved.
	DisapprovedCountries []string `json:"disapprovedCountries,omitempty"`
	// PendingCountries: List of country codes (ISO 3166-1 alpha-2) where the offer
	// is pending approval.
	PendingCountries []string `json:"pendingCountries,omitempty"`
	// ReportingContext: The name of the reporting context.
	//
	// Possible values:
	//   "REPORTING_CONTEXT_ENUM_UNSPECIFIED" - Not specified.
	//   "SHOPPING_ADS" - [Shopping
	// ads](https://support.google.com/merchants/answer/6149970).
	//   "DISCOVERY_ADS" - Deprecated: Use `DEMAND_GEN_ADS` instead. [Discovery and
	// Demand Gen ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS" - [Demand Gen
	// ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS_DISCOVER_SURFACE" - [Demand Gen ads on Discover
	// surface](https://support.google.com/merchants/answer/13389785).
	//   "VIDEO_ADS" - [Video
	// ads](https://support.google.com/google-ads/answer/6340491).
	//   "DISPLAY_ADS" - [Display
	// ads](https://support.google.com/merchants/answer/6069387).
	//   "LOCAL_INVENTORY_ADS" - [Local inventory
	// ads](https://support.google.com/merchants/answer/3271956).
	//   "VEHICLE_INVENTORY_ADS" - [Vehicle inventory
	// ads](https://support.google.com/merchants/answer/11544533).
	//   "FREE_LISTINGS" - [Free product
	// listings](https://support.google.com/merchants/answer/9199328).
	//   "FREE_LOCAL_LISTINGS" - [Free local product
	// listings](https://support.google.com/merchants/answer/9825611).
	//   "FREE_LOCAL_VEHICLE_LISTINGS" - [Free local vehicle
	// listings](https://support.google.com/merchants/answer/11544533).
	//   "YOUTUBE_SHOPPING" - [YouTube
	// Shopping](https://support.google.com/merchants/answer/13478370).
	//   "CLOUD_RETAIL" - [Cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "LOCAL_CLOUD_RETAIL" - [Local cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "PRODUCT_REVIEWS" - [Product
	// Reviews](https://support.google.com/merchants/answer/14620732).
	//   "MERCHANT_REVIEWS" - [Merchant
	// Reviews](https://developers.google.com/merchant-review-feeds).
	//   "YOUTUBE_CHECKOUT" - [YouTube Checkout](
	ReportingContext string `json:"reportingContext,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ApprovedCountries") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ApprovedCountries") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DestinationStatus) MarshalJSON() ([]byte, error) {
	type NoMethod DestinationStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Empty: A generic empty message that you can re-use to avoid defining
// duplicated empty messages in your APIs. A typical example is to use it as
// the request or the response type of an API method. For instance: service Foo
// { rpc Bar(google.protobuf.Empty) returns (google.protobuf.Empty); }
type Empty struct {
	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
}

// FreeShippingThreshold: Conditions to be met for a product to have free
// shipping.
type FreeShippingThreshold struct {
	// Country: The CLDR territory code
	// (http://www.unicode.org/repos/cldr/tags/latest/common/main/en.xml) of the
	// country to which an item will ship.
	Country string `json:"country,omitempty"`
	// PriceThreshold: The minimum product price for the shipping cost to become
	// free. Represented as a number.
	PriceThreshold *Price `json:"priceThreshold,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Country") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Country") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FreeShippingThreshold) MarshalJSON() ([]byte, error) {
	type NoMethod FreeShippingThreshold
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Installment: A message that represents installment.
type Installment struct {
	// Amount: The amount the buyer has to pay per month.
	Amount *Price `json:"amount,omitempty"`
	// CreditType: Type of installment payments. Supported values are: *
	// "finance" * "lease"
	CreditType string `json:"creditType,omitempty"`
	// Downpayment: The up-front down payment amount the buyer has to pay.
	Downpayment *Price `json:"downpayment,omitempty"`
	// Months: The number of installments the buyer has to pay.
	Months int64 `json:"months,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "Amount") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Amount") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Installment) MarshalJSON() ([]byte, error) {
	type NoMethod Installment
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Interval: Represents a time interval, encoded as a Timestamp start
// (inclusive) and a Timestamp end (exclusive). The start must be less than or
// equal to the end. When the start equals the end, the interval is empty
// (matches no time). When both start and end are unspecified, the interval
// matches any time.
type Interval struct {
	// EndTime: Optional. Exclusive end of the interval. If specified, a Timestamp
	// matching this interval will have to be before the end.
	EndTime string `json:"endTime,omitempty"`
	// StartTime: Optional. Inclusive start of the interval. If specified, a
	// Timestamp matching this interval will have to be the same or after the
	// start.
	StartTime string `json:"startTime,omitempty"`
	// ForceSendFields is a list of field names (e.g. "EndTime") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "EndTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Interval) MarshalJSON() ([]byte, error) {
	type NoMethod Interval
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ItemLevelIssue: The ItemLevelIssue of the product status.
type ItemLevelIssue struct {
	// ApplicableCountries: List of country codes (ISO 3166-1 alpha-2) where issue
	// applies to the offer.
	ApplicableCountries []string `json:"applicableCountries,omitempty"`
	// Attribute: The attribute's name, if the issue is caused by a single
	// attribute.
	Attribute string `json:"attribute,omitempty"`
	// Code: The error code of the issue.
	Code string `json:"code,omitempty"`
	// Description: A short issue description in English.
	Description string `json:"description,omitempty"`
	// Detail: A detailed issue description in English.
	Detail string `json:"detail,omitempty"`
	// Documentation: The URL of a web page to help with resolving this issue.
	Documentation string `json:"documentation,omitempty"`
	// ReportingContext: The reporting context the issue applies to.
	//
	// Possible values:
	//   "REPORTING_CONTEXT_ENUM_UNSPECIFIED" - Not specified.
	//   "SHOPPING_ADS" - [Shopping
	// ads](https://support.google.com/merchants/answer/6149970).
	//   "DISCOVERY_ADS" - Deprecated: Use `DEMAND_GEN_ADS` instead. [Discovery and
	// Demand Gen ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS" - [Demand Gen
	// ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS_DISCOVER_SURFACE" - [Demand Gen ads on Discover
	// surface](https://support.google.com/merchants/answer/13389785).
	//   "VIDEO_ADS" - [Video
	// ads](https://support.google.com/google-ads/answer/6340491).
	//   "DISPLAY_ADS" - [Display
	// ads](https://support.google.com/merchants/answer/6069387).
	//   "LOCAL_INVENTORY_ADS" - [Local inventory
	// ads](https://support.google.com/merchants/answer/3271956).
	//   "VEHICLE_INVENTORY_ADS" - [Vehicle inventory
	// ads](https://support.google.com/merchants/answer/11544533).
	//   "FREE_LISTINGS" - [Free product
	// listings](https://support.google.com/merchants/answer/9199328).
	//   "FREE_LOCAL_LISTINGS" - [Free local product
	// listings](https://support.google.com/merchants/answer/9825611).
	//   "FREE_LOCAL_VEHICLE_LISTINGS" - [Free local vehicle
	// listings](https://support.google.com/merchants/answer/11544533).
	//   "YOUTUBE_SHOPPING" - [YouTube
	// Shopping](https://support.google.com/merchants/answer/13478370).
	//   "CLOUD_RETAIL" - [Cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "LOCAL_CLOUD_RETAIL" - [Local cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "PRODUCT_REVIEWS" - [Product
	// Reviews](https://support.google.com/merchants/answer/14620732).
	//   "MERCHANT_REVIEWS" - [Merchant
	// Reviews](https://developers.google.com/merchant-review-feeds).
	//   "YOUTUBE_CHECKOUT" - [YouTube Checkout](
	ReportingContext string `json:"reportingContext,omitempty"`
	// Resolution: Whether the issue can be resolved by the merchant.
	Resolution string `json:"resolution,omitempty"`
	// Severity: How this issue affects serving of the offer.
	//
	// Possible values:
	//   "SEVERITY_UNSPECIFIED" - Not specified.
	//   "NOT_IMPACTED" - This issue represents a warning and does not have a
	// direct affect on the product.
	//   "DEMOTED" - The product is demoted and most likely have limited
	// performance in search results
	//   "DISAPPROVED" - Issue disapproves the product.
	Severity string `json:"severity,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ApplicableCountries") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ApplicableCountries") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ItemLevelIssue) MarshalJSON() ([]byte, error) {
	type NoMethod ItemLevelIssue
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListProductsResponse: Response message for the ListProducts method.
type ListProductsResponse struct {
	// NextPageToken: A token, which can be sent as `page_token` to retrieve the
	// next page. If this field is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// Products: The processed products from the specified account. These are your
	// processed products after applying rules and supplemental data sources.
	Products []*Product `json:"products,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListProductsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListProductsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LoyaltyPoints: A message that represents loyalty points.
type LoyaltyPoints struct {
	// Name: Name of loyalty points program. It is recommended to limit the name to
	// 12 full-width characters or 24 Roman characters.
	Name string `json:"name,omitempty"`
	// PointsValue: The retailer's loyalty points in absolute value.
	PointsValue int64 `json:"pointsValue,omitempty,string"`
	// Ratio: The ratio of a point when converted to currency. Google assumes
	// currency based on Merchant Center settings. If ratio is left out, it
	// defaults to 1.0.
	Ratio float64 `json:"ratio,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Name") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Name") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LoyaltyPoints) MarshalJSON() ([]byte, error) {
	type NoMethod LoyaltyPoints
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *LoyaltyPoints) UnmarshalJSON(data []byte) error {
	type NoMethod LoyaltyPoints
	var s1 struct {
		Ratio gensupport.JSONFloat64 `json:"ratio"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Ratio = float64(s1.Ratio)
	return nil
}

// LoyaltyProgram: A message that represents loyalty program.
type LoyaltyProgram struct {
	// CashbackForFutureUse: The cashback that can be used for future purchases.
	CashbackForFutureUse *Price `json:"cashbackForFutureUse,omitempty"`
	// LoyaltyPoints: The amount of loyalty points earned on a purchase.
	LoyaltyPoints int64 `json:"loyaltyPoints,omitempty,string"`
	// MemberPriceEffectiveDate: A date range during which the item is eligible for
	// member price. If not specified, the member price is always applicable. The
	// date range is represented by a pair of ISO 8601 dates separated by a space,
	// comma, or slash.
	MemberPriceEffectiveDate *Interval `json:"memberPriceEffectiveDate,omitempty"`
	// Price: The price for members of the given tier, that is, the instant
	// discount price. Must be smaller or equal to the regular price.
	Price *Price `json:"price,omitempty"`
	// ProgramLabel: The label of the loyalty program. This is an internal label
	// that uniquely identifies the relationship between a merchant entity and a
	// loyalty program entity. The label must be provided so that the system can
	// associate the assets below (for example, price and points) with a merchant.
	// The corresponding program must be linked to the merchant account.
	ProgramLabel string `json:"programLabel,omitempty"`
	// ShippingLabel: The label of the shipping benefit. If the field has value,
	// this offer has loyalty shipping benefit. If the field value isn't provided,
	// the item is not eligible for loyalty shipping for the given loyalty tier.
	ShippingLabel string `json:"shippingLabel,omitempty"`
	// TierLabel: The label of the tier within the loyalty program. Must match one
	// of the labels within the program.
	TierLabel string `json:"tierLabel,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CashbackForFutureUse") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CashbackForFutureUse") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LoyaltyProgram) MarshalJSON() ([]byte, error) {
	type NoMethod LoyaltyProgram
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Price: The price represented as a number and currency.
type Price struct {
	// AmountMicros: The price represented as a number in micros (1 million micros
	// is an equivalent to one's currency standard unit, for example, 1 USD =
	// 1000000 micros).
	AmountMicros int64 `json:"amountMicros,omitempty,string"`
	// CurrencyCode: The currency of the price using three-letter acronyms
	// according to ISO 4217 (http://en.wikipedia.org/wiki/ISO_4217).
	CurrencyCode string `json:"currencyCode,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AmountMicros") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AmountMicros") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Price) MarshalJSON() ([]byte, error) {
	type NoMethod Price
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Product: The processed product, built from multiple product inputs after
// applying rules and supplemental data sources. This processed product matches
// what is shown in your Merchant Center account and in Shopping ads and other
// surfaces across Google. Each product is built from exactly one primary data
// source product input, and multiple supplemental data source inputs. After
// inserting, updating, or deleting a product input, it may take several
// minutes before the updated processed product can be retrieved. All fields in
// the processed product and its sub-messages match the name of their
// corresponding attribute in the Product data specification
// (https://support.google.com/merchants/answer/7052112) with some exceptions.
type Product struct {
	// Attributes: Output only. A list of product attributes.
	Attributes *Attributes `json:"attributes,omitempty"`
	// Channel: Output only. The channel
	// (https://support.google.com/merchants/answer/7361332) of the product.
	//
	// Possible values:
	//   "CHANNEL_ENUM_UNSPECIFIED" - Not specified.
	//   "ONLINE" - Online product.
	//   "LOCAL" - Local product.
	Channel string `json:"channel,omitempty"`
	// ContentLanguage: Output only. The two-letter ISO 639-1
	// (http://en.wikipedia.org/wiki/ISO_639-1) language code for the product.
	ContentLanguage string `json:"contentLanguage,omitempty"`
	// CustomAttributes: Output only. A list of custom (merchant-provided)
	// attributes. It can also be used to submit any attribute of the data
	// specification in its generic form (for example, `{ "name": "size type",
	// "value": "regular" }`). This is useful for submitting attributes not
	// explicitly exposed by the API, such as additional attributes used for Buy on
	// Google.
	CustomAttributes []*CustomAttribute `json:"customAttributes,omitempty"`
	// DataSource: Output only. The primary data source of the product.
	DataSource string `json:"dataSource,omitempty"`
	// FeedLabel: Output only. The feed label for the product.
	FeedLabel string `json:"feedLabel,omitempty"`
	// Name: The name of the product. Format:
	// "{product.name=accounts/{account}/products/{product}}" where the last
	// section `product` consists of 4 parts:
	// channel~content_language~feed_label~offer_id example for product name is
	// "accounts/123/products/online~en~US~sku123"
	Name string `json:"name,omitempty"`
	// OfferId: Output only. Your unique identifier for the product. This is the
	// same for the product input and processed product. Leading and trailing
	// whitespaces are stripped and multiple whitespaces are replaced by a single
	// whitespace upon submission. See the product data specification
	// (https://support.google.com/merchants/answer/188494#id) for details.
	OfferId string `json:"offerId,omitempty"`
	// ProductStatus: Output only. The status of a product, data validation issues,
	// that is, information about a product computed asynchronously.
	ProductStatus *ProductStatus `json:"productStatus,omitempty"`
	// VersionNumber: Output only. Represents the existing version (freshness) of
	// the product, which can be used to preserve the right order when multiple
	// updates are done at the same time. If set, the insertion is prevented when
	// version number is lower than the current version number of the existing
	// product. Re-insertion (for example, product refresh after 30 days) can be
	// performed with the current `version_number`. Only supported for insertions
	// into primary data sources. If the operation is prevented, the aborted
	// exception will be thrown.
	VersionNumber int64 `json:"versionNumber,omitempty,string"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Attributes") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Attributes") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Product) MarshalJSON() ([]byte, error) {
	type NoMethod Product
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductChange: The change that happened to the product including old value,
// new value, country code as the region code and reporting context.
type ProductChange struct {
	// NewValue: The new value of the changed resource or attribute.
	NewValue string `json:"newValue,omitempty"`
	// OldValue: The old value of the changed resource or attribute.
	OldValue string `json:"oldValue,omitempty"`
	// RegionCode: Countries that have the change (if applicable)
	RegionCode string `json:"regionCode,omitempty"`
	// ReportingContext: Reporting contexts that have the change (if applicable)
	//
	// Possible values:
	//   "REPORTING_CONTEXT_ENUM_UNSPECIFIED" - Not specified.
	//   "SHOPPING_ADS" - [Shopping
	// ads](https://support.google.com/merchants/answer/6149970).
	//   "DISCOVERY_ADS" - Deprecated: Use `DEMAND_GEN_ADS` instead. [Discovery and
	// Demand Gen ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS" - [Demand Gen
	// ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS_DISCOVER_SURFACE" - [Demand Gen ads on Discover
	// surface](https://support.google.com/merchants/answer/13389785).
	//   "VIDEO_ADS" - [Video
	// ads](https://support.google.com/google-ads/answer/6340491).
	//   "DISPLAY_ADS" - [Display
	// ads](https://support.google.com/merchants/answer/6069387).
	//   "LOCAL_INVENTORY_ADS" - [Local inventory
	// ads](https://support.google.com/merchants/answer/3271956).
	//   "VEHICLE_INVENTORY_ADS" - [Vehicle inventory
	// ads](https://support.google.com/merchants/answer/11544533).
	//   "FREE_LISTINGS" - [Free product
	// listings](https://support.google.com/merchants/answer/9199328).
	//   "FREE_LOCAL_LISTINGS" - [Free local product
	// listings](https://support.google.com/merchants/answer/9825611).
	//   "FREE_LOCAL_VEHICLE_LISTINGS" - [Free local vehicle
	// listings](https://support.google.com/merchants/answer/11544533).
	//   "YOUTUBE_SHOPPING" - [YouTube
	// Shopping](https://support.google.com/merchants/answer/13478370).
	//   "CLOUD_RETAIL" - [Cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "LOCAL_CLOUD_RETAIL" - [Local cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "PRODUCT_REVIEWS" - [Product
	// Reviews](https://support.google.com/merchants/answer/14620732).
	//   "MERCHANT_REVIEWS" - [Merchant
	// Reviews](https://developers.google.com/merchant-review-feeds).
	//   "YOUTUBE_CHECKOUT" - [YouTube Checkout](
	ReportingContext string `json:"reportingContext,omitempty"`
	// ForceSendFields is a list of field names (e.g. "NewValue") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NewValue") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductChange) MarshalJSON() ([]byte, error) {
	type NoMethod ProductChange
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductDetail: The product details.
type ProductDetail struct {
	// AttributeName: The name of the product detail.
	AttributeName string `json:"attributeName,omitempty"`
	// AttributeValue: The value of the product detail.
	AttributeValue string `json:"attributeValue,omitempty"`
	// SectionName: The section header used to group a set of product details.
	SectionName string `json:"sectionName,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AttributeName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AttributeName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductDetail) MarshalJSON() ([]byte, error) {
	type NoMethod ProductDetail
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductDimension: The dimension of the product.
type ProductDimension struct {
	// Unit: Required. The dimension units. Acceptable values are: * "in" *
	// "cm"
	Unit string `json:"unit,omitempty"`
	// Value: Required. The dimension value represented as a number. The value can
	// have a maximum precision of four decimal places.
	Value float64 `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Unit") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Unit") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductDimension) MarshalJSON() ([]byte, error) {
	type NoMethod ProductDimension
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *ProductDimension) UnmarshalJSON(data []byte) error {
	type NoMethod ProductDimension
	var s1 struct {
		Value gensupport.JSONFloat64 `json:"value"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Value = float64(s1.Value)
	return nil
}

// ProductInput: This resource represents input data you submit for a product,
// not the processed product that you see in Merchant Center, in Shopping ads,
// or across Google surfaces. Product inputs, rules and supplemental data
// source data are combined to create the processed Product. Required product
// input attributes to pass data validation checks are primarily defined in the
// Products Data Specification
// (https://support.google.com/merchants/answer/188494). The following
// attributes are required: feedLabel, contentLanguage and offerId. After
// inserting, updating, or deleting a product input, it may take several
// minutes before the processed product can be retrieved. All fields in the
// product input and its sub-messages match the English name of their
// corresponding attribute in the vertical spec with some exceptions
// (https://support.google.com/merchants/answer/7052112).
type ProductInput struct {
	// Attributes: Optional. A list of product attributes.
	Attributes *Attributes `json:"attributes,omitempty"`
	// Channel: Required. Immutable. The channel
	// (https://support.google.com/merchants/answer/7361332) of the product.
	//
	// Possible values:
	//   "CHANNEL_ENUM_UNSPECIFIED" - Not specified.
	//   "ONLINE" - Online product.
	//   "LOCAL" - Local product.
	Channel string `json:"channel,omitempty"`
	// ContentLanguage: Required. Immutable. The two-letter ISO 639-1
	// (http://en.wikipedia.org/wiki/ISO_639-1) language code for the product.
	ContentLanguage string `json:"contentLanguage,omitempty"`
	// CustomAttributes: Optional. A list of custom (merchant-provided) attributes.
	// It can also be used for submitting any attribute of the data specification
	// in its generic form (for example, `{ "name": "size type", "value": "regular"
	// }`). This is useful for submitting attributes not explicitly exposed by the
	// API, such as additional attributes used for Buy on Google. Maximum allowed
	// number of characters for each custom attribute is 10240 (represents sum of
	// characters for name and value). Maximum 2500 custom attributes can be set
	// per product, with total size of 102.4kB. Underscores in custom attribute
	// names are replaced by spaces upon insertion.
	CustomAttributes []*CustomAttribute `json:"customAttributes,omitempty"`
	// FeedLabel: Required. Immutable. The feed label
	// (https://developers.google.com/shopping-content/guides/products/feed-labels)
	// for the product.
	FeedLabel string `json:"feedLabel,omitempty"`
	// Name: Identifier. The name of the product input. Format:
	// "{productinput.name=accounts/{account}/productInputs/{productinput}}"
	// where the last section `productinput` consists of 4 parts:
	// channel~content_language~feed_label~offer_id example for product input name
	// is "accounts/123/productInputs/online~en~US~sku123"
	Name string `json:"name,omitempty"`
	// OfferId: Required. Immutable. Your unique identifier for the product. This
	// is the same for the product input and processed product. Leading and
	// trailing whitespaces are stripped and multiple whitespaces are replaced by a
	// single whitespace upon submission. See the products data specification
	// (https://support.google.com/merchants/answer/188494#id) for details.
	OfferId string `json:"offerId,omitempty"`
	// Product: Output only. The name of the processed product. Format:
	// "{product.name=accounts/{account}/products/{product}}"
	Product string `json:"product,omitempty"`
	// VersionNumber: Optional. Represents the existing version (freshness) of the
	// product, which can be used to preserve the right order when multiple updates
	// are done at the same time. If set, the insertion is prevented when version
	// number is lower than the current version number of the existing product.
	// Re-insertion (for example, product refresh after 30 days) can be performed
	// with the current `version_number`. Only supported for insertions into
	// primary data sources. If the operation is prevented, the aborted exception
	// will be thrown.
	VersionNumber int64 `json:"versionNumber,omitempty,string"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Attributes") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Attributes") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductInput) MarshalJSON() ([]byte, error) {
	type NoMethod ProductInput
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductStatus: The status of a product, data validation issues, that is,
// information about a product computed asynchronously.
type ProductStatus struct {
	// CreationDate: Date on which the item has been created, in ISO 8601
	// (http://en.wikipedia.org/wiki/ISO_8601) format.
	CreationDate string `json:"creationDate,omitempty"`
	// DestinationStatuses: The intended destinations for the product.
	DestinationStatuses []*DestinationStatus `json:"destinationStatuses,omitempty"`
	// GoogleExpirationDate: Date on which the item expires, in ISO 8601
	// (http://en.wikipedia.org/wiki/ISO_8601) format.
	GoogleExpirationDate string `json:"googleExpirationDate,omitempty"`
	// ItemLevelIssues: A list of all issues associated with the product.
	ItemLevelIssues []*ItemLevelIssue `json:"itemLevelIssues,omitempty"`
	// LastUpdateDate: Date on which the item has been last updated, in ISO 8601
	// (http://en.wikipedia.org/wiki/ISO_8601) format.
	LastUpdateDate string `json:"lastUpdateDate,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CreationDate") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CreationDate") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductStatus) MarshalJSON() ([]byte, error) {
	type NoMethod ProductStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductStatusChangeMessage: The message that the merchant will receive to
// notify about product status change event
type ProductStatusChangeMessage struct {
	// Account: The target account that owns the entity that changed. Format :
	// `accounts/{merchant_id}`
	Account string `json:"account,omitempty"`
	// Attribute: The attribute in the resource that changed, in this case it will
	// be always `Status`.
	//
	// Possible values:
	//   "ATTRIBUTE_UNSPECIFIED" - Unspecified attribute
	//   "STATUS" - Status of the changed entity
	Attribute string `json:"attribute,omitempty"`
	// Changes: A message to describe the change that happened to the product
	Changes []*ProductChange `json:"changes,omitempty"`
	// ExpirationTime: The product expiration time. This field will not bet set if
	// the notification is sent for a product deletion event.
	ExpirationTime string `json:"expirationTime,omitempty"`
	// ManagingAccount: The account that manages the merchant's account. can be the
	// same as merchant id if it is standalone account. Format :
	// `accounts/{service_provider_id}`
	ManagingAccount string `json:"managingAccount,omitempty"`
	// Resource: The product name. Format:
	// `{product.name=accounts/{account}/products/{product}}`
	Resource string `json:"resource,omitempty"`
	// ResourceId: The product id.
	ResourceId string `json:"resourceId,omitempty"`
	// ResourceType: The resource that changed, in this case it will always be
	// `Product`.
	//
	// Possible values:
	//   "RESOURCE_UNSPECIFIED" - Unspecified resource
	//   "PRODUCT" - Resource type : product
	ResourceType string `json:"resourceType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Account") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Account") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductStatusChangeMessage) MarshalJSON() ([]byte, error) {
	type NoMethod ProductStatusChangeMessage
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductStructuredDescription: Structured description, for algorithmically
// (AI)-generated descriptions.
type ProductStructuredDescription struct {
	// Content: The description text Maximum length is 5000 characters
	Content string `json:"content,omitempty"`
	// DigitalSourceType: The digital source type, for example
	// "trained_algorithmic_media". Following IPTC
	// (https://cv.iptc.org/newscodes/digitalsourcetype). Maximum length is 40
	// characters.
	DigitalSourceType string `json:"digitalSourceType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Content") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Content") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductStructuredDescription) MarshalJSON() ([]byte, error) {
	type NoMethod ProductStructuredDescription
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductStructuredTitle: Structured title, for algorithmically (AI)-generated
// titles.
type ProductStructuredTitle struct {
	// Content: The title text Maximum length is 150 characters
	Content string `json:"content,omitempty"`
	// DigitalSourceType: The digital source type, for example
	// "trained_algorithmic_media". Following IPTC
	// (https://cv.iptc.org/newscodes/digitalsourcetype). Maximum length is 40
	// characters.
	DigitalSourceType string `json:"digitalSourceType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Content") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Content") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductStructuredTitle) MarshalJSON() ([]byte, error) {
	type NoMethod ProductStructuredTitle
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductSustainabilityIncentive: Information regarding sustainability-related
// incentive programs such as rebates or tax relief.
type ProductSustainabilityIncentive struct {
	// Amount: The fixed amount of the incentive.
	Amount *Price `json:"amount,omitempty"`
	// Percentage: The percentage of the sale price that the incentive is applied
	// to.
	Percentage float64 `json:"percentage,omitempty"`
	// Type: Sustainability incentive program.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Unspecified or unknown sustainability incentive type.
	//   "EV_TAX_CREDIT" - Program offering tax liability reductions for electric
	// vehicles and, in some countries, plug-in hybrids. These reductions can be
	// based on a specific amount or a percentage of the sale price.
	//   "EV_PRICE_DISCOUNT" - A subsidy program, often called an environmental
	// bonus, provides a purchase grant for electric vehicles and, in some
	// countries, plug-in hybrids. The grant amount may be a fixed sum or a
	// percentage of the sale price.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Amount") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Amount") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductSustainabilityIncentive) MarshalJSON() ([]byte, error) {
	type NoMethod ProductSustainabilityIncentive
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *ProductSustainabilityIncentive) UnmarshalJSON(data []byte) error {
	type NoMethod ProductSustainabilityIncentive
	var s1 struct {
		Percentage gensupport.JSONFloat64 `json:"percentage"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Percentage = float64(s1.Percentage)
	return nil
}

// ProductWeight: The weight of the product.
type ProductWeight struct {
	// Unit: Required. The weight unit. Acceptable values are: * "g" * "kg" *
	// "oz" * "lb"
	Unit string `json:"unit,omitempty"`
	// Value: Required. The weight represented as a number. The weight can have a
	// maximum precision of four decimal places.
	Value float64 `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Unit") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Unit") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductWeight) MarshalJSON() ([]byte, error) {
	type NoMethod ProductWeight
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *ProductWeight) UnmarshalJSON(data []byte) error {
	type NoMethod ProductWeight
	var s1 struct {
		Value gensupport.JSONFloat64 `json:"value"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Value = float64(s1.Value)
	return nil
}

// Shipping: The Shipping of the product.
type Shipping struct {
	// Country: The CLDR territory code
	// (http://www.unicode.org/repos/cldr/tags/latest/common/main/en.xml) of the
	// country to which an item will ship.
	Country string `json:"country,omitempty"`
	// LocationGroupName: The location where the shipping is applicable,
	// represented by a location group name.
	LocationGroupName string `json:"locationGroupName,omitempty"`
	// LocationId: The numeric ID of a location that the shipping rate applies to
	// as defined in the AdWords API
	// (https://developers.google.com/adwords/api/docs/appendix/geotargeting).
	LocationId int64 `json:"locationId,omitempty,string"`
	// MaxHandlingTime: Maximum handling time (inclusive) between when the order is
	// received and shipped in business days. 0 means that the order is shipped on
	// the same day as it is received if it happens before the cut-off time. Both
	// maxHandlingTime and maxTransitTime are required if providing shipping
	// speeds. minHandlingTime is optional if maxHandlingTime is present.
	MaxHandlingTime int64 `json:"maxHandlingTime,omitempty,string"`
	// MaxTransitTime: Maximum transit time (inclusive) between when the order has
	// shipped and when it is delivered in business days. 0 means that the order is
	// delivered on the same day as it ships. Both maxHandlingTime and
	// maxTransitTime are required if providing shipping speeds. minTransitTime is
	// optional if maxTransitTime is present.
	MaxTransitTime int64 `json:"maxTransitTime,omitempty,string"`
	// MinHandlingTime: Minimum handling time (inclusive) between when the order is
	// received and shipped in business days. 0 means that the order is shipped on
	// the same day as it is received if it happens before the cut-off time.
	// minHandlingTime can only be present together with maxHandlingTime; but it is
	// not required if maxHandlingTime is present.
	MinHandlingTime int64 `json:"minHandlingTime,omitempty,string"`
	// MinTransitTime: Minimum transit time (inclusive) between when the order has
	// shipped and when it is delivered in business days. 0 means that the order is
	// delivered on the same day as it ships. minTransitTime can only be present
	// together with maxTransitTime; but it is not required if maxTransitTime is
	// present.
	MinTransitTime int64 `json:"minTransitTime,omitempty,string"`
	// PostalCode: The postal code range that the shipping rate applies to,
	// represented by a postal code, a postal code prefix followed by a * wildcard,
	// a range between two postal codes or two postal code prefixes of equal
	// length.
	PostalCode string `json:"postalCode,omitempty"`
	// Price: Fixed shipping price, represented as a number.
	Price *Price `json:"price,omitempty"`
	// Region: The geographic region to which a shipping rate applies. See region
	// (https://support.google.com/merchants/answer/6324484) for more information.
	Region string `json:"region,omitempty"`
	// Service: A free-form description of the service class or delivery speed.
	Service string `json:"service,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Country") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Country") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Shipping) MarshalJSON() ([]byte, error) {
	type NoMethod Shipping
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ShippingDimension: The ShippingDimension of the product.
type ShippingDimension struct {
	// Unit: The unit of value.
	Unit string `json:"unit,omitempty"`
	// Value: The dimension of the product used to calculate the shipping cost of
	// the item.
	Value float64 `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Unit") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Unit") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ShippingDimension) MarshalJSON() ([]byte, error) {
	type NoMethod ShippingDimension
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *ShippingDimension) UnmarshalJSON(data []byte) error {
	type NoMethod ShippingDimension
	var s1 struct {
		Value gensupport.JSONFloat64 `json:"value"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Value = float64(s1.Value)
	return nil
}

// ShippingWeight: The ShippingWeight of the product.
type ShippingWeight struct {
	// Unit: The unit of value.
	Unit string `json:"unit,omitempty"`
	// Value: The weight of the product used to calculate the shipping cost of the
	// item.
	Value float64 `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Unit") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Unit") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ShippingWeight) MarshalJSON() ([]byte, error) {
	type NoMethod ShippingWeight
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *ShippingWeight) UnmarshalJSON(data []byte) error {
	type NoMethod ShippingWeight
	var s1 struct {
		Value gensupport.JSONFloat64 `json:"value"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Value = float64(s1.Value)
	return nil
}

// SubscriptionCost: The SubscriptionCost of the product.
type SubscriptionCost struct {
	// Amount: The amount the buyer has to pay per subscription period.
	Amount *Price `json:"amount,omitempty"`
	// Period: The type of subscription period. Supported values are: * "month" *
	// "year"
	//
	// Possible values:
	//   "SUBSCRIPTION_PERIOD_UNSPECIFIED" - Indicates that the subscription period
	// is unspecified.
	//   "MONTH" - Indicates that the subscription period is month.
	//   "YEAR" - Indicates that the subscription period is year.
	Period string `json:"period,omitempty"`
	// PeriodLength: The number of subscription periods the buyer has to pay.
	PeriodLength int64 `json:"periodLength,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "Amount") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Amount") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SubscriptionCost) MarshalJSON() ([]byte, error) {
	type NoMethod SubscriptionCost
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Tax: The Tax of the product.
type Tax struct {
	// Country: The country within which the item is taxed, specified as a CLDR
	// territory code
	// (http://www.unicode.org/repos/cldr/tags/latest/common/main/en.xml).
	Country string `json:"country,omitempty"`
	// LocationId: The numeric ID of a location that the tax rate applies to as
	// defined in the AdWords API
	// (https://developers.google.com/adwords/api/docs/appendix/geotargeting).
	LocationId int64 `json:"locationId,omitempty,string"`
	// PostalCode: The postal code range that the tax rate applies to, represented
	// by a ZIP code, a ZIP code prefix using * wildcard, a range between two ZIP
	// codes or two ZIP code prefixes of equal length. Examples: 94114, 94*,
	// 94002-95460, 94*-95*.
	PostalCode string `json:"postalCode,omitempty"`
	// Rate: The percentage of tax rate that applies to the item price.
	Rate float64 `json:"rate,omitempty"`
	// Region: The geographic region to which the tax rate applies.
	Region string `json:"region,omitempty"`
	// TaxShip: Set to true if tax is charged on shipping.
	TaxShip bool `json:"taxShip,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Country") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Country") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Tax) MarshalJSON() ([]byte, error) {
	type NoMethod Tax
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *Tax) UnmarshalJSON(data []byte) error {
	type NoMethod Tax
	var s1 struct {
		Rate gensupport.JSONFloat64 `json:"rate"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Rate = float64(s1.Rate)
	return nil
}

// UnitPricingBaseMeasure: The UnitPricingBaseMeasure of the product.
type UnitPricingBaseMeasure struct {
	// Unit: The unit of the denominator.
	Unit string `json:"unit,omitempty"`
	// Value: The denominator of the unit price.
	Value int64 `json:"value,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "Unit") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Unit") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s UnitPricingBaseMeasure) MarshalJSON() ([]byte, error) {
	type NoMethod UnitPricingBaseMeasure
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// UnitPricingMeasure: The UnitPricingMeasure of the product.
type UnitPricingMeasure struct {
	// Unit: The unit of the measure.
	Unit string `json:"unit,omitempty"`
	// Value: The measure of an item.
	Value float64 `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Unit") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Unit") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s UnitPricingMeasure) MarshalJSON() ([]byte, error) {
	type NoMethod UnitPricingMeasure
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *UnitPricingMeasure) UnmarshalJSON(data []byte) error {
	type NoMethod UnitPricingMeasure
	var s1 struct {
		Value gensupport.JSONFloat64 `json:"value"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Value = float64(s1.Value)
	return nil
}

type AccountsProductInputsDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a product input from your Merchant Center account. After
// inserting, updating, or deleting a product input, it may take several
// minutes before the processed product can be retrieved.
//
//   - name: The name of the product input resource to delete. Format:
//     accounts/{account}/productInputs/{product} where the last section
//     `product` consists of 4 parts:
//     channel~content_language~feed_label~offer_id example for product name is
//     "accounts/123/productInputs/online~en~US~sku123".
func (r *AccountsProductInputsService) Delete(name string) *AccountsProductInputsDeleteCall {
	c := &AccountsProductInputsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// DataSource sets the optional parameter "dataSource": Required. The primary
// or supplemental data source from which the product input should be deleted.
// Format: `accounts/{account}/dataSources/{datasource}`.
func (c *AccountsProductInputsDeleteCall) DataSource(dataSource string) *AccountsProductInputsDeleteCall {
	c.urlParams_.Set("dataSource", dataSource)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsProductInputsDeleteCall) Fields(s ...googleapi.Field) *AccountsProductInputsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsProductInputsDeleteCall) Context(ctx context.Context) *AccountsProductInputsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsProductInputsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsProductInputsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "products/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.productInputs.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.productInputs.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsProductInputsDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.productInputs.delete", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsProductInputsInsertCall struct {
	s            *Service
	parent       string
	productinput *ProductInput
	urlParams_   gensupport.URLParams
	ctx_         context.Context
	header_      http.Header
}

// Insert: Uploads a product input to your Merchant Center account. If an input
// with the same contentLanguage, offerId, and dataSource already exists, this
// method replaces that entry. After inserting, updating, or deleting a product
// input, it may take several minutes before the processed product can be
// retrieved.
//
//   - parent: The account where this product will be inserted. Format:
//     accounts/{account}.
func (r *AccountsProductInputsService) Insert(parent string, productinput *ProductInput) *AccountsProductInputsInsertCall {
	c := &AccountsProductInputsInsertCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.productinput = productinput
	return c
}

// DataSource sets the optional parameter "dataSource": Required. The primary
// or supplemental product data source name. If the product already exists and
// data source provided is different, then the product will be moved to a new
// data source. Format: `accounts/{account}/dataSources/{datasource}`.
func (c *AccountsProductInputsInsertCall) DataSource(dataSource string) *AccountsProductInputsInsertCall {
	c.urlParams_.Set("dataSource", dataSource)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsProductInputsInsertCall) Fields(s ...googleapi.Field) *AccountsProductInputsInsertCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsProductInputsInsertCall) Context(ctx context.Context) *AccountsProductInputsInsertCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsProductInputsInsertCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsProductInputsInsertCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.productinput)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "products/v1beta/{+parent}/productInputs:insert")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.productInputs.insert", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.productInputs.insert" call.
// Any non-2xx status code is an error. Response headers are in either
// *ProductInput.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsProductInputsInsertCall) Do(opts ...googleapi.CallOption) (*ProductInput, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ProductInput{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.productInputs.insert", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsProductsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Retrieves the processed product from your Merchant Center account.
// After inserting, updating, or deleting a product input, it may take several
// minutes before the updated final product can be retrieved.
//
//   - name: The name of the product to retrieve. Format:
//     `accounts/{account}/products/{product}` where the last section `product`
//     consists of 4 parts: channel~content_language~feed_label~offer_id example
//     for product name is "accounts/123/products/online~en~US~sku123".
func (r *AccountsProductsService) Get(name string) *AccountsProductsGetCall {
	c := &AccountsProductsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsProductsGetCall) Fields(s ...googleapi.Field) *AccountsProductsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsProductsGetCall) IfNoneMatch(entityTag string) *AccountsProductsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsProductsGetCall) Context(ctx context.Context) *AccountsProductsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsProductsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsProductsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "products/v1beta/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.products.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.products.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Product.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsProductsGetCall) Do(opts ...googleapi.CallOption) (*Product, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Product{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.products.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsProductsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists the processed products in your Merchant Center account. The
// response might contain fewer items than specified by pageSize. Rely on
// pageToken to determine if there are more items to be requested. After
// inserting, updating, or deleting a product input, it may take several
// minutes before the updated processed product can be retrieved.
//
//   - parent: The account to list processed products for. Format:
//     accounts/{account}.
func (r *AccountsProductsService) List(parent string) *AccountsProductsListCall {
	c := &AccountsProductsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// products to return. The service may return fewer than this value. The
// maximum value is 250; values above 250 will be coerced to 250. If
// unspecified, the maximum number of products will be returned.
func (c *AccountsProductsListCall) PageSize(pageSize int64) *AccountsProductsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListProducts` call. Provide this to retrieve the subsequent
// page. When paginating, all other parameters provided to `ListProducts` must
// match the call that provided the page token.
func (c *AccountsProductsListCall) PageToken(pageToken string) *AccountsProductsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsProductsListCall) Fields(s ...googleapi.Field) *AccountsProductsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsProductsListCall) IfNoneMatch(entityTag string) *AccountsProductsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsProductsListCall) Context(ctx context.Context) *AccountsProductsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsProductsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsProductsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "products/v1beta/{+parent}/products")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.products.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.products.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListProductsResponse.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsProductsListCall) Do(opts ...googleapi.CallOption) (*ListProductsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListProductsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.products.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsProductsListCall) Pages(ctx context.Context, f func(*ListProductsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}
