#!/bin/bash
#
# Download Go, verify its signature, and if that all succeeds, move the tarball
# to go.tar.gz in the current directory.

set -eu

if [ $# -eq 0 ]; then
  echo "usage: $0 <go version> [platform like linux-amd64]"
  exit 1
fi
VERSION="${1}"
PLATFORM="${2:-linux-amd64}"

export GNUPGHOME="$(mktemp -d)"

# From https://www.google.com/linuxrepositories/
#
# Key Details
#   Download: https://dl.google.com/linux/linux_signing_key.pub
#   Key ID: Google, Inc. Linux Package Signing Key <linux-packages-keymaster@google.com>
#   Fingerprint: 4CCA 1EAF 950C EE4A B839 76DC A040 830F 7FAC 5991
#   Google, Inc. (Linux Package Signing Authority) <linux-packages-keymaster@google.com>
#   Fingerprint: EB4C 1BFD 4F04 2F6D DDCC EC91 7721 F63B D38B 4796
gpg2 --import <<EOF
-----BEGIN PGP PUBLIC KEY BLOCK-----
Version: GnuPG v1.4.2.2 (GNU/Linux)

mQGiBEXwb0YRBADQva2NLpYXxgjNkbuP0LnPoEXruGmvi3XMIxjEUFuGNCP4Rj/a
kv2E5VixBP1vcQFDRJ+p1puh8NU0XERlhpyZrVMzzS/RdWdyXf7E5S8oqNXsoD1z
fvmI+i9b2EhHAA19Kgw7ifV8vMa4tkwslEmcTiwiw8lyUl28Wh4Et8SxzwCggDcA
feGqtn3PP5YAdD0km4S4XeMEAJjlrqPoPv2Gf//tfznY2UyS9PUqFCPLHgFLe80u
QhI2U5jt6jUKN4fHauvR6z3seSAsh1YyzyZCKxJFEKXCCqnrFSoh4WSJsbFNc4PN
b0V0SqiTCkWADZyLT5wll8sWuQ5ylTf3z1ENoHf+G3um3/wk/+xmEHvj9HCTBEXP
78X0A/0Tqlhc2RBnEf+AqxWvM8sk8LzJI/XGjwBvKfXe+l3rnSR2kEAvGzj5Sg0X
4XmfTg4Jl8BNjWyvm2Wmjfet41LPmYJKsux3g0b8yzQxeOA4pQKKAU3Z4+rgzGmf
HdwCG5MNT2A5XxD/eDd+L4fRx0HbFkIQoAi1J3YWQSiTk15fw7RMR29vZ2xlLCBJ
bmMuIExpbnV4IFBhY2thZ2UgU2lnbmluZyBLZXkgPGxpbnV4LXBhY2thZ2VzLWtl
eW1hc3RlckBnb29nbGUuY29tPohjBBMRAgAjAhsDBgsJCAcDAgQVAggDBBYCAwEC
HgECF4AFAkYVdn8CGQEACgkQoECDD3+sWZHKSgCfdq3HtNYJLv+XZleb6HN4zOcF
AJEAniSFbuv8V5FSHxeRimHx25671az+uQINBEXwb0sQCACuA8HT2nr+FM5y/kzI
A51ZcC46KFtIDgjQJ31Q3OrkYP8LbxOpKMRIzvOZrsjOlFmDVqitiVc7qj3lYp6U
rgNVaFv6Qu4bo2/ctjNHDDBdv6nufmusJUWq/9TwieepM/cwnXd+HMxu1XBKRVk9
XyAZ9SvfcW4EtxVgysI+XlptKFa5JCqFM3qJllVohMmr7lMwO8+sxTWTXqxsptJo
pZeKz+UBEEqPyw7CUIVYGC9ENEtIMFvAvPqnhj1GS96REMpry+5s9WKuLEaclWpd
K3krttbDlY1NaeQUCRvBYZ8iAG9YSLHUHMTuI2oea07Rh4dtIAqPwAX8xn36JAYG
2vgLAAMFB/wKqaycjWAZwIe98Yt0qHsdkpmIbarD9fGiA6kfkK/UxjL/k7tmS4Vm
CljrrDZkPSQ/19mpdRcGXtb0NI9+nyM5trweTvtPw+HPkDiJlTaiCcx+izg79Fj9
KcofuNb3lPdXZb9tzf5oDnmm/B+4vkeTuEZJ//IFty8cmvCpzvY+DAz1Vo9rA+Zn
cpWY1n6z6oSS9AsyT/IFlWWBZZ17SpMHu+h4Bxy62+AbPHKGSujEGQhWq8ZRoJAT
G0KSObnmZ7FwFWu1e9XFoUCt0bSjiJWTIyaObMrWu/LvJ3e9I87HseSJStfw6fki
5og9qFEkMrIrBCp3QGuQWBq/rTdMuwNFiEkEGBECAAkFAkXwb0sCGwwACgkQoECD
D3+sWZF/WACfeNAu1/1hwZtUo1bR+MWiCjpvHtwAnA1R3IHqFLQ2X3xJ40XPuAyY
/FJG
=Quqp
-----END PGP PUBLIC KEY BLOCK-----
-----BEGIN PGP PUBLIC KEY BLOCK-----
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=ilBW
-----END PGP PUBLIC KEY BLOCK-----
EOF


OUTPUT_DIR="$(mktemp -d)"
FILENAME="go${VERSION}.${PLATFORM}.tar.gz"
OUTPUT_FILE="${OUTPUT_DIR}/${FILENAME}"
curl -sSL "https://dl.google.com/go/${FILENAME}.asc" -o "${OUTPUT_FILE}.asc"
curl -sSL "https://dl.google.com/go/${FILENAME}" -o "${OUTPUT_FILE}"

gpg2 --verify --trust-model=always "${OUTPUT_FILE}.asc" "${OUTPUT_FILE}"

mv "${OUTPUT_FILE}" go.tar.gz

rm -r "${OUTPUT_DIR}"
rm -r "${GNUPGHOME}"
