package io

import (
	"encoding/json"
	"fmt"
	"io"
	"maps"
	"slices"

	"git.sr.ht/~charles/rq/util"
	"github.com/alecthomas/chroma/v2/quick"
	"github.com/hashicorp/hcl/v2/hclwrite"
	"github.com/zclconf/go-cty/cty"
)

// This file is substantially based on codec/hcl_codec.go from
// https://github.com/JFryy/qq
//
// The original license of this file is included below. Modifications are
// covered by the rq project license.
//
///////////////////////////////////////////////////////////////////////////////
//
// MIT License
//
// Copyright (c) 2024 JFry
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
//
///////////////////////////////////////////////////////////////////////////////

func init() {
	registerOutputHandler("hcl", func() OutputHandler { return &HCLOutputHandler{} })
}

// Declare conformance with OutputHandler interface.
var _ OutputHandler = &HCLOutputHandler{}

// HCLOutputHandler handles serializing HCL data.
type HCLOutputHandler struct {
	style       string
	colorize    bool
	initialized bool
}

func (h *HCLOutputHandler) init() {
	if h.initialized {
		return
	}

	h.colorize = true
	h.style = "native"
	h.initialized = true
}

// Name implements OutputHandler.Name()
func (h *HCLOutputHandler) Name() string {
	return "hcl"
}

// SetOption implements OutputHandler.SetOption().
func (h *HCLOutputHandler) SetOption(name string, value string) error {
	h.init()

	switch name {
	case "output.colorize":
		h.colorize = util.StringToValue(value).(bool)
	case "output.style":
		h.style = value
	}

	return nil
}

// Format implements OutputHandler.Format()
func (h *HCLOutputHandler) Format(writer io.Writer, data interface{}) error {
	h.init()

	hclBytes, err := hclMarshal(data)
	if err != nil {
		return err
	}

	if h.colorize {
		err := quick.Highlight(writer, string(hclBytes), "hcl", "terminal", h.style)
		if err != nil {
			return err
		}
	} else {
		_, err := writer.Write(hclBytes)
		if err != nil {
			return err
		}
	}

	return nil

}

func hclMarshal(v interface{}) ([]byte, error) {
	// Ensure the input is wrapped in a map if it's not already
	var data map[string]interface{}
	switch v := v.(type) {
	case map[string]interface{}:
		data = v
	default:
		data = map[string]interface{}{
			"data": v,
		}
	}

	// Convert map to HCL
	hclData, err := convertMapToHCL(data)
	if err != nil {
		return nil, fmt.Errorf("error converting map to HCL: %w", err)
	}

	return hclData, nil
}

func convertMapToHCL(data map[string]interface{}) ([]byte, error) {
	// Create a new HCL file
	f := hclwrite.NewEmptyFile()

	// Create the root body of the file
	rootBody := f.Body()

	// Populate the body with the data
	err := populateBody(rootBody, data)
	if err != nil {
		return nil, err
	}

	// Return the HCL data
	return f.Bytes(), nil
}

func populateBody(body *hclwrite.Body, data map[string]interface{}) error {
	for _, key := range slices.Sorted(maps.Keys(data)) {
		value := data[key]
		switch v := value.(type) {
		case map[string]interface{}:
			block := body.AppendNewBlock(key, nil)
			err := populateBody(block.Body(), v)
			if err != nil {
				return err
			}
		case string:
			body.SetAttributeValue(key, cty.StringVal(v))
		case int:
			body.SetAttributeValue(key, cty.NumberIntVal(int64(v)))
		case int64:
			body.SetAttributeValue(key, cty.NumberIntVal(v))
		case float64:
			body.SetAttributeValue(key, cty.NumberFloatVal(v))
		case json.Number:
			f, err := v.Float64()
			if err != nil {
				return err
			}
			body.SetAttributeValue(key, cty.NumberFloatVal(f))
		case bool:
			body.SetAttributeValue(key, cty.BoolVal(v))
		case []interface{}:
			tuple := make([]cty.Value, len(v))
			var err error
			for i, elem := range v {
				tuple[i], err = convertToCtyValue(elem)
				if err != nil {
					return err
				}
			}
			body.SetAttributeValue(key, cty.TupleVal(tuple))
		case []map[string]interface{}:
			tuple := make([]cty.Value, len(v))
			var err error
			for i, elem := range v {
				tuple[i], err = convertToCtyValue(elem)
				if err != nil {
					return err
				}
			}
			body.SetAttributeValue(key, cty.TupleVal(tuple))
		default:
			return fmt.Errorf("Unsupported type: %T", v)
		}
	}
	return nil
}

func convertToCtyValue(value interface{}) (cty.Value, error) {
	switch v := value.(type) {
	case string:
		return cty.StringVal(v), nil
	case int:
		return cty.NumberIntVal(int64(v)), nil
	case int64:
		return cty.NumberIntVal(v), nil
	case float64:
		return cty.NumberFloatVal(v), nil
	case json.Number:
		f, err := v.Float64()
		if err != nil {
			return cty.NilVal, err
		}
		return cty.NumberFloatVal(f), nil
	case bool:
		return cty.BoolVal(v), nil
	case []interface{}:
		tuple := make([]cty.Value, len(v))
		var err error
		for i, elem := range v {
			tuple[i], err = convertToCtyValue(elem)
			if err != nil {
				return cty.NilVal, err
			}
		}
		return cty.TupleVal(tuple), nil
	case []map[string]interface{}:
		tuple := make([]cty.Value, len(v))
		var err error
		for i, elem := range v {
			tuple[i], err = convertToCtyValue(elem)
			if err != nil {
				return cty.NilVal, err
			}
		}
		return cty.TupleVal(tuple), nil
	case map[string]interface{}:
		vals := make(map[string]cty.Value)
		var err error
		for _, k := range slices.Sorted(maps.Keys(v)) {
			elem := v[k]
			vals[k], err = convertToCtyValue(elem)
			if err != nil {
				return cty.NilVal, err
			}
		}
		return cty.ObjectVal(vals), nil
	default:
		return cty.NilVal, fmt.Errorf("Unsupported type: %T", v)
	}
}
