/* 
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.portlet.proxy;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;

import javax.portlet.ActionResponse;
import javax.portlet.ClientDataRequest;
import javax.portlet.MimeResponse;
import javax.portlet.PortletRequest;
import javax.portlet.PortletResponse;
import javax.servlet.http.Cookie;

import org.apache.portals.applications.webcontent2.proxy.RequestContext;
import org.apache.portals.applications.webcontent2.rewriter.Sink;

/**
 * {@link javax.portlet.PortletRequest} / {@link javax.portlet.PortletResponse}
 * abstraction needed for reverse proxy processing.
 */
public class PortletRequestContext implements RequestContext
{

    /**
     * Internal portlet request.
     */
    private final PortletRequest request;

    /**
     * Internal portlet response.
     */
    private final PortletResponse response;

    /**
     * Response status holder.
     */
    private int responseStatus;

    /**
     * Response cookies holder.
     */
    private List<Cookie> responseCookies;

    /**
     * Portlet response based {@link Sink}.
     */
    private Sink sink;

    /**
     * Constructs a {@link PortletRequestContext} with given portlet request and portlet response.
     * @param request
     * @param response
     */
    public PortletRequestContext(final PortletRequest request, final PortletResponse response)
    {
        this.request = request;
        this.response = response;
    }

    /**
     * Returns the underlying portlet request.
     * @return
     */
    public PortletRequest getPortletRequest()
    {
        return request;
    }

    /**
     * Returns the underlying portlet response.
     * @return
     */
    public PortletResponse getPortletResponse()
    {
        return response;
    }

    /**
     * {@inheritDoc}
     */
    public boolean isSecure()
    {
        return request.isSecure();
    }

    /**
     * {@inheritDoc}
     */
    public String getScheme()
    {
        return request.getScheme();
    }

    /**
     * {@inheritDoc}
     */
    public String getServerName()
    {
        return request.getServerName();
    }

    /**
     * {@inheritDoc}
     */
    public int getServerPort()
    {
        return request.getServerPort();
    }

    /**
     * {@inheritDoc}
     */
    public String getMethod()
    {
        if (request instanceof ClientDataRequest)
        {
            return ((ClientDataRequest) request).getMethod();
        }

        return "GET";
    }

    /**
     * {@inheritDoc}
     */
    public String getRequestBasePath()
    {
        return "";
    }

    /**
     * {@inheritDoc}
     */
    public String getPathInfo()
    {
        return "/";
    }

    /**
     * {@inheritDoc}
     */
    public Object getAttribute(String name)
    {
        return request.getAttribute(name);
    }

    /**
     * {@inheritDoc}
     */
    public String getQueryString()
    {
        return null;
    }

    /**
     * {@inheritDoc}
     */
    public Enumeration getHeaderNames()
    {
        return request.getPropertyNames();
    }

    /**
     * {@inheritDoc}
     */
    public Enumeration getHeaders(String name)
    {
        return request.getProperties(name);
    }

    /**
     * {@inheritDoc}
     */
    public String getHeader(String name)
    {
        return request.getProperty(name);
    }

    /**
     * {@inheritDoc}
     */
    public InputStream getInputStream() throws IOException
    {
        if (request instanceof ClientDataRequest)
        {
            return ((ClientDataRequest) request).getPortletInputStream();
        }

        return null;
    }

    /**
     * {@inheritDoc}
     */
    public void setStatus(int sc)
    {
        this.responseStatus = sc;
    }

    public int getStatus()
    {
        return this.responseStatus;
    }

    /**
     * {@inheritDoc}
     */
    public void sendRedirect(String location) throws IOException
    {
        if (response instanceof ActionResponse)
        {
            ((ActionResponse) response).sendRedirect(location);
        }
    }

    /**
     * {@inheritDoc}
     */
    public void setHeader(String name, String value)
    {
        response.setProperty(name, value);
    }

    /**
     * {@inheritDoc}
     */
    public void setIntHeader(String name, int value)
    {
        response.setProperty(name, Integer.toString(value));
    }

    /**
     * {@inheritDoc}
     */
    public void addCookie(Cookie cookie)
    {
        if (responseCookies == null)
        {
            responseCookies = new ArrayList<Cookie>();
        }

        responseCookies.add(cookie);
    }

    /**
     * {@inheritDoc}
     */
    public List<Cookie> getCookies()
    {
        if (responseCookies == null)
        {
            return Collections.emptyList();
        }

        return Collections.unmodifiableList(responseCookies);
    }

    /**
     * {@inheritDoc}
     */
    public Sink getSink()
    {
        if (sink == null)
        {
            if (!(response instanceof MimeResponse))
            {
                throw new IllegalStateException("MimeResponse is required to create a sink.");
            }

            sink = new MimeResponseSink((MimeResponse) response);
        }

        return sink;
    }
}
