/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.rewriter;

import static org.junit.Assert.assertTrue;

import java.io.StringReader;
import java.io.StringWriter;
import java.net.URI;
import java.util.List;

import org.apache.commons.io.IOUtils;
import org.apache.portals.applications.webcontent2.proxy.ProxyContext;
import org.apache.portals.applications.webcontent2.proxy.ProxyMapping;
import org.apache.portals.applications.webcontent2.proxy.ProxyMappingRegistry;
import org.apache.portals.applications.webcontent2.proxy.impl.DefaultProxyMappingRegistry;
import org.apache.portals.applications.webcontent2.proxy.impl.RegexProxyMapping;
import org.apache.portals.applications.webcontent2.proxy.impl.ServletRequestContext;
import org.apache.portals.applications.webcontent2.proxy.impl.SimpleProxyMapping;
import org.apache.portals.applications.webcontent2.rewriter.ContentRewriter;
import org.apache.portals.applications.webcontent2.rewriter.ContentRewritingContext;
import org.apache.portals.applications.webcontent2.rewriter.Sink;
import org.apache.portals.applications.webcontent2.rewriter.Source;
import org.apache.portals.applications.webcontent2.rewriter.impl.SimpleContentRewritingContext;
import org.apache.portals.applications.webcontent2.rewriter.impl.StreamSink;
import org.apache.portals.applications.webcontent2.rewriter.impl.StreamSource;
import org.junit.Before;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultReverseProxyTextLineContentRewriterTest
{

    private static Logger log = LoggerFactory.getLogger(DefaultReverseProxyTextLineContentRewriterTest.class);

    private static final String BASE_REQUEST_PATH = "/app1/rproxyservlet";
    private static final String BASE_PORTALS_LOCAL_PATH = "/portals/";
    private static final String BASE_REMOTE_URI = "http://portals.apache.org/";
    private static final String CURRENT_BASE_PATHINFO = "some/path/";

    private ContentRewritingContext rewritingContext;

    private ContentRewriter contentRewriter;
    private String htmlSource1;

    @Before
    public void before() throws Exception
    {
        rewritingContext = new SimpleContentRewritingContext();

        ServletRequestContext requestContext = new ServletRequestContext(null, null);
        requestContext.setRequestBasePath(BASE_REQUEST_PATH);
        ProxyContext proxyContext = new ProxyContext(requestContext);
        proxyContext.setRemoteURI(URI.create(BASE_REMOTE_URI + CURRENT_BASE_PATHINFO + "page.html"));

        DefaultProxyMappingRegistry registry = new DefaultProxyMappingRegistry();

        SimpleProxyMapping currentMapping = new SimpleProxyMapping();
        currentMapping.setLocal(BASE_PORTALS_LOCAL_PATH);
        currentMapping.setRemote(URI.create(BASE_REMOTE_URI));
        registry.addProxyMapping(currentMapping);

        RegexProxyMapping regexMapping = new RegexProxyMapping();
        regexMapping.setLocalPattern("^/apache/([^/]+)/(.*)$");
        regexMapping.setRemoteReplace("http://$1.apache.org/$2");
        regexMapping.setRemotePattern("^http://(\\w+)\\.apache\\.org/(.*)$");
        regexMapping.setLocalReplace("/apache/$1/$2");
        registry.addProxyMapping(regexMapping);

        SimpleProxyMapping simpleMapping = new SimpleProxyMapping();
        simpleMapping.setLocal("/apache/");
        simpleMapping.setRemote(URI.create("http://apache.org/"));
        registry.addProxyMapping(simpleMapping);

        rewritingContext.setAttribute(ProxyContext.class.getName(), proxyContext);
        rewritingContext.setAttribute(ProxyMapping.class.getName(), currentMapping);
        rewritingContext.setAttribute(ProxyMappingRegistry.class.getName(), registry);

        contentRewriter = new DefaultReverseProxyTextLineContentRewriter();
        htmlSource1 = IOUtils.toString(getClass().getResource("content-rewriter-test1.html"));
    }

    @Test
    public void testSimple() throws Exception {
        Source source = new StreamSource(new StringReader(htmlSource1));
        StringWriter sw = new StringWriter();
        Sink sink = new StreamSink(sw);

        contentRewriter.rewrite(source, sink, rewritingContext);

        log.debug("SINK: {}", sw);

        List<String> sourceLines = IOUtils.readLines(new StringReader(htmlSource1));
        List<String> rewrittenLines = IOUtils.readLines(new StringReader(sw.toString()));

        int lineNum = 21;
        assertTrue("Wrong rewriting: \n\t" + sourceLines.get(lineNum) + "\n\t" + rewrittenLines.get(lineNum), 
                   rewrittenLines.get(lineNum).contains("href=\"" + BASE_REQUEST_PATH + BASE_PORTALS_LOCAL_PATH + "skin/css/screen1.css\""));
        lineNum = 22;
        assertTrue("Wrong rewriting: \n\t" + sourceLines.get(lineNum) + "\n\t" + rewrittenLines.get(lineNum), 
                   rewrittenLines.get(lineNum).contains("href=\"" + BASE_REQUEST_PATH + BASE_PORTALS_LOCAL_PATH + "skin/css/screen2.css\""));
        lineNum = 24;
        assertTrue("Wrong rewriting: \n\t" + sourceLines.get(lineNum) + "\n\t" + rewrittenLines.get(lineNum), 
                   rewrittenLines.get(lineNum).contains("url('" + BASE_REQUEST_PATH + BASE_PORTALS_LOCAL_PATH + "skin/css/import1.css');"));
        lineNum = 25;
        assertTrue("Wrong rewriting: \n\t" + sourceLines.get(lineNum) + "\n\t" + rewrittenLines.get(lineNum), 
                   rewrittenLines.get(lineNum).contains("url( \"" + BASE_REQUEST_PATH + BASE_PORTALS_LOCAL_PATH + "skin/css/import2.css\" );"));
        lineNum = 29;
        assertTrue("Wrong rewriting: \n\t" + sourceLines.get(lineNum) + "\n\t" + rewrittenLines.get(lineNum), 
                   rewrittenLines.get(lineNum).contains("href=\"" + BASE_REQUEST_PATH + BASE_PORTALS_LOCAL_PATH + CURRENT_BASE_PATHINFO + "mission.html\""));
        assertTrue("Wrong rewriting: \n\t" + sourceLines.get(lineNum) + "\n\t" + rewrittenLines.get(lineNum), 
                   rewrittenLines.get(lineNum).contains("href='" + BASE_REQUEST_PATH + BASE_PORTALS_LOCAL_PATH + CURRENT_BASE_PATHINFO + "about.html'"));
        lineNum = 30;
        assertTrue("Wrong rewriting: \n\t" + sourceLines.get(lineNum) + "\n\t" + rewrittenLines.get(lineNum), 
                   rewrittenLines.get(lineNum).contains("href=\"" + BASE_REQUEST_PATH + BASE_PORTALS_LOCAL_PATH + "docs/\""));
        assertTrue("Wrong rewriting: \n\t" + sourceLines.get(lineNum) + "\n\t" + rewrittenLines.get(lineNum), 
                   rewrittenLines.get(lineNum).contains("href='" + BASE_REQUEST_PATH + BASE_PORTALS_LOCAL_PATH + "documents/'"));
        lineNum = 31;
        assertTrue("Wrong rewriting: \n\t" + sourceLines.get(lineNum) + "\n\t" + rewrittenLines.get(lineNum), 
                   rewrittenLines.get(lineNum).contains("href=\"" + BASE_REQUEST_PATH + "/apache/projects/\""));
        assertTrue("Wrong rewriting: \n\t" + sourceLines.get(lineNum) + "\n\t" + rewrittenLines.get(lineNum), 
                   rewrittenLines.get(lineNum).contains("href=\"" + BASE_REQUEST_PATH + BASE_PORTALS_LOCAL_PATH + "apa.html\""));
        lineNum = 32;
        assertTrue("Wrong rewriting: \n\t" + sourceLines.get(lineNum) + "\n\t" + rewrittenLines.get(lineNum), 
                   rewrittenLines.get(lineNum).contains("href=\"" + BASE_REQUEST_PATH + "/apache/www/events.html\""));
        assertTrue("Wrong rewriting: \n\t" + sourceLines.get(lineNum) + "\n\t" + rewrittenLines.get(lineNum), 
                   rewrittenLines.get(lineNum).contains("href='" + BASE_REQUEST_PATH + "/apache/www/apachecon.html'"));
        lineNum = 33;
        assertTrue("Wrong rewriting: \n\t" + sourceLines.get(lineNum) + "\n\t" + rewrittenLines.get(lineNum), 
                   rewrittenLines.get(lineNum).contains("href=\"http://www.example.com/manager/list\""));
        assertTrue("Wrong rewriting: \n\t" + sourceLines.get(lineNum) + "\n\t" + rewrittenLines.get(lineNum), 
                   rewrittenLines.get(lineNum).contains("href='http://www.example.com/manager/start'"));
        lineNum = 35;
        assertTrue("Wrong rewriting: \n\t" + sourceLines.get(lineNum) + "\n\t" + rewrittenLines.get(lineNum), 
                   rewrittenLines.get(lineNum).contains("href=\"" + BASE_REQUEST_PATH + BASE_PORTALS_LOCAL_PATH + "docs/\""));
        assertTrue("Wrong rewriting: \n\t" + sourceLines.get(lineNum) + "\n\t" + rewrittenLines.get(lineNum), 
                   rewrittenLines.get(lineNum).contains("href='" + BASE_REQUEST_PATH + BASE_PORTALS_LOCAL_PATH + "documents/'"));
        lineNum = 37;
        assertTrue("Wrong rewriting: \n\t" + sourceLines.get(lineNum) + "\n\t" + rewrittenLines.get(lineNum), 
                   rewrittenLines.get(lineNum).contains("href=\"http://www.example.com/manager/list\""));
        lineNum = 38;
        assertTrue("Wrong rewriting: \n\t" + sourceLines.get(lineNum) + "\n\t" + rewrittenLines.get(lineNum), 
                   rewrittenLines.get(lineNum).contains("href=\"https://blogs.apache.org/index.html\""));
        lineNum = 39;
        assertTrue("Wrong rewriting: \n\t" + sourceLines.get(lineNum) + "\n\t" + rewrittenLines.get(lineNum), 
                   rewrittenLines.get(lineNum).contains("href=\"" + BASE_REQUEST_PATH + "/apache/index.html\""));
        lineNum = 40;
        assertTrue("Wrong rewriting: \n\t" + sourceLines.get(lineNum) + "\n\t" + rewrittenLines.get(lineNum), 
                   rewrittenLines.get(lineNum).contains("<script>new AjaxUpdate('/lazyLoader');</script>"));
    }
}
