/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.io.Closeable;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.chain.Catalog;
import org.apache.commons.chain.Chain;
import org.apache.commons.chain.Command;
import org.apache.commons.chain.impl.CatalogBase;
import org.apache.portals.applications.webcontent2.proxy.impl.AbstractProxyCommand;
import org.apache.portals.applications.webcontent2.proxy.impl.ProxyProcessingChain;
import org.junit.Before;
import org.junit.Test;

/**
 * Test case to verify commons-chain is good for the use cases in this project.
 * <P>
 * It's necessary to assemble multiple valve components in a pipeline,
 * with each valve component being able to stop or proceed,
 * the valves being able to be organized in a hierarchical way,
 * in different ways such as simple configuration, spring beans assembly or programmatic way.
 * </P>
 * <P>
 * Also, it is not demonstrated here, but one more good thing with commons-chain is that
 * you can add implement <code>Filter</code> in a command (valve) object to invoke post-processing
 * individually.
 * </P>
 */
public class ProxyCommandChainTest
{

    private Catalog catalog;

    /**
     * The <code>catalog</code> is set up as equivalent as the following here:
     * <br/>
     * <xmp>
     * <catalog>
     *   <chain name="default">
     *     <chain name="preprocessing">
     *       <command className="...InitCommand"/>
     *     </chain>
     *     <chain name="processing">
     *       <command className="...AuthCommand"/>
     *       <command className="...RenderingCommand"/>
     *     </chain>
     *     <chain name="postprocessing">
     *       <command className="...CleanupCommand"/>
     *     </chain>
     *   </chain>
     * </catalog>
     * </xmp>
     * @throws Exception
     */
    @Before
    public void before() throws Exception 
    {
        Map<String, Command> commandsMap = new HashMap<String, Command>();

        Command initCommand = new InitCommand();
        Command authCommand = new AuthCommand();
        Command renderingCommand = new RenderingCommand();
        Command notInvokedCommand = new NotInvokedCommand();
        Command cleanupCommand = new CleanupCommand();

        Chain preprocessingChain = new ProxyProcessingChain(new Command [] { initCommand });
        Chain processingChain = new ProxyProcessingChain(new Command [] { authCommand, renderingCommand, notInvokedCommand });
        Chain postprocessingChain = new ProxyProcessingChain(new Command [] { cleanupCommand });

        Chain defaultChain = new ProxyProcessingChain(new Command [] { preprocessingChain, processingChain, postprocessingChain });

        commandsMap.put("default", defaultChain);

        catalog = new CatalogBase(commandsMap);
    }

    @SuppressWarnings("unchecked")
    @Test
    public void testDefaultChainWithoutAuthorization() throws Exception 
    {
        Command defaultCommand = catalog.getCommand("default");

        ProxyContext context = new ProxyContext(null);
        StringWriter sw = new StringWriter();
        context.put("out", new PrintWriter(sw));
        context.put("user.name", "John");

        defaultCommand.execute(context);

        assertNull(context.get("error.code"));
        List<String> logs = (List<String>) context.get("logs");
        assertTrue(logs.contains("InitCommand is about to execute."));
        assertTrue(logs.contains("InitCommand has executed."));
        assertTrue(logs.contains("AuthCommand is about to execute."));
        assertTrue(logs.contains("AuthCommand has executed."));
        assertTrue(logs.contains("RenderingCommand is about to execute."));
        assertTrue(logs.contains("RenderingCommand has executed."));
        assertFalse(logs.contains("NotInvokedCommand is about to execute."));
        assertFalse(logs.contains("NotInvokedCommand has executed."));
        assertTrue(logs.contains("CleanupCommand is about to execute."));
        assertTrue(logs.contains("CleanupCommand has executed."));
        assertEquals("Hello, John!", sw.toString());
    }

    @SuppressWarnings("unchecked")
    @Test
    public void testDefaultChainWithAuthorization() throws Exception 
    {
        Command defaultCommand = catalog.getCommand("default");

        ProxyContext context = new ProxyContext(null);
        StringWriter sw = new StringWriter();
        context.put("out", new PrintWriter(sw));
        context.put("user.name", "John");
        context.put("role.allowed", "manager");
        context.put("user.roles", new HashSet<String>(Arrays.asList("user", "manager")));

        defaultCommand.execute(context);

        assertNull(context.get("error.code"));
        List<String> logs = (List<String>) context.get("logs");
        assertTrue(logs.contains("InitCommand is about to execute."));
        assertTrue(logs.contains("InitCommand has executed."));
        assertTrue(logs.contains("AuthCommand is about to execute."));
        assertTrue(logs.contains("AuthCommand has executed."));
        assertTrue(logs.contains("RenderingCommand is about to execute."));
        assertTrue(logs.contains("RenderingCommand has executed."));
        assertFalse(logs.contains("NotInvokedCommand is about to execute."));
        assertFalse(logs.contains("NotInvokedCommand has executed."));
        assertTrue(logs.contains("CleanupCommand is about to execute."));
        assertTrue(logs.contains("CleanupCommand has executed."));
        assertEquals("Hello, John!", sw.toString());
    }

    @SuppressWarnings("unchecked")
    @Test
    public void testDefaultChainWithAuthorizationFailure() throws Exception 
    {
        Command defaultCommand = catalog.getCommand("default");

        ProxyContext context = new ProxyContext(null);
        StringWriter sw = new StringWriter();
        context.put("out", new PrintWriter(sw));
        context.put("user.name", "John");
        context.put("role.allowed", "manager");
        context.put("user.roles", new HashSet<String>(Arrays.asList("user", "poweruser")));

        defaultCommand.execute(context);

        assertEquals(Integer.valueOf(403), context.get("error.code"));
        List<String> logs = (List<String>) context.get("logs");
        assertTrue(logs.contains("InitCommand is about to execute."));
        assertTrue(logs.contains("InitCommand has executed."));
        assertTrue(logs.contains("AuthCommand is about to execute."));
        assertTrue(logs.contains("AuthCommand has executed."));
        assertFalse(logs.contains("RenderingCommand is about to execute."));
        assertFalse(logs.contains("RenderingCommand has executed."));
        assertFalse(logs.contains("NotInvokedCommand is about to execute."));
        assertFalse(logs.contains("NotInvokedCommand has executed."));
        assertTrue(logs.contains("CleanupCommand is about to execute."));
        assertTrue(logs.contains("CleanupCommand has executed."));
        assertEquals("", sw.toString());
    }

    private abstract class AbstractLoggableCommand extends AbstractProxyCommand
    {
        @SuppressWarnings("unchecked")
        protected boolean executeInternal(ProxyContext context) throws ReverseProxyException, IOException
        {
            try {
                addLog(context, " is about to execute.");
                return executeInternal2(context);
            } finally {
                addLog(context, " has executed.");
            }
        }

        protected void addLog(final ProxyContext context, final String message) {
            List<String> logs = (List<String>) context.get("logs");

            if (logs == null) {
                logs = new LinkedList<String>();
                context.put("logs", logs);
            }

            logs.add(getClass().getSimpleName() + message);
        }

        protected abstract boolean executeInternal2(ProxyContext context) throws ReverseProxyException, IOException;
    }

    private class InitCommand extends AbstractLoggableCommand
    {
        @Override
        protected boolean executeInternal2(ProxyContext context) throws ReverseProxyException, IOException
        {
            return false;
        }
    }

    private class AuthCommand extends AbstractLoggableCommand
    {
        @SuppressWarnings("unchecked")
        @Override
        protected boolean executeInternal2(ProxyContext context) throws ReverseProxyException, IOException
        {
            final String roleAllowed = (String) context.get("role.allowed");

            if (roleAllowed == null) 
            {
                // no authorization; so continue.
                return false;
            }

            Set<String> userRoles = (Set<String>) context.get("user.roles");

            if (userRoles == null || !userRoles.contains(roleAllowed))
            {
                // authorization failed; stop processing with error info
                context.put("error.code", Integer.valueOf(403));
                return true;
            }

            // continue processing
            return false;
        }
    }

    private class RenderingCommand extends AbstractLoggableCommand
    {
        @Override
        protected boolean executeInternal2(ProxyContext context) throws ReverseProxyException, IOException
        {
            PrintWriter out = (PrintWriter) context.get("out");
            out.print("Hello, " + context.get("user.name") + "!");
            out.flush();

            return true;
        }
    }

    private class NotInvokedCommand extends AbstractLoggableCommand
    {
        @Override
        protected boolean executeInternal2(ProxyContext context) throws ReverseProxyException, IOException
        {
            return false;
        }
    }

    private class CleanupCommand extends AbstractLoggableCommand
    {
        @SuppressWarnings("unchecked")
        @Override
        protected boolean executeInternal2(ProxyContext context) throws ReverseProxyException, IOException
        {
            Collection<Closeable> closeables = (Collection<Closeable>) context.get("closeables");

            if (closeables != null) {
                for (Closeable closeable : closeables)
                {
                    closeable.close();
                }
            }

            return false;
        }
    }

}
